{

    Unit CoverDownloads

    Class TCoverDownloadWorkerThread
        A Workerthread for downloading Covers from LastFM using its API

        ---------------------------------------------------------------
    Covernator
    Copyright (C) 2010, Daniel Gaussmann
    http://www.gausi.de
    mail@gausi.de
    ---------------------------------------------------------------
    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA 02110, USA
    ---------------------------------------------------------------

}
unit CoverDownloads;

interface

uses
  Windows, Messages, SysUtils,  Classes, Graphics,
  Dialogs, StrUtils, ContNrs, Jpeg, PNGImage, DateUtils,
  IdBaseComponent, IdComponent, IdHTTP, IdStack, IdException,
  CoverHelper;

const
    // Api_Key vom Mp3FileUtils-Demo-Account
    Api_Key = 'f7fa672081fb6bf5a1e9ae618dcd4fdf';

type
    TPicType = (ptNone, ptJPG, ptPNG);

    TCoverDownloader = class;

    TCoverDownloaderStatusEvent = procedure(Sender: TCoverDownloader; Status: String) of Object;
    TCoverDownloaderBeforeSaveEvent = procedure(Sender: TCoverDownloader; PicType: TPicType; var Filename: String) of Object;


    TCoverDownloader = class
        private
            { Private-Deklarationen }
            fIDHttp: TIdHttp;
            fDataStream: TMemoryStream;  // Stream containing the Picture-Data
            fDataType: TPicType;
            fSavedFile: String;

            fOnXMLComplete   : TCoverDownloaderStatusEvent;
            fOnCoverURLFound : TCoverDownloaderStatusEvent;
            fOnError         : TCoverDownloaderStatusEvent;
            fOnBeforeSave    : TCoverDownloaderBeforeSaveEvent;

            function QueryLastFMCoverXML(aArtist, aAlbum: String): AnsiString;
            function GetBestCoverUrlFromXML(aXMLData: AnsiString): AnsiString;
            function DownloadBestCoverToStream(aCoverURL: AnsiString): Boolean;

            function SavePicStreamToFile(aDirectory: String): Boolean;
        public
            constructor Create;
            destructor Destroy; override;

            property OnXMLComplete   : TCoverDownloaderStatusEvent       read fOnXMLComplete    write fOnXMLComplete   ;
            property OnCoverURLFound : TCoverDownloaderStatusEvent       read fOnCoverURLFound  write fOnCoverURLFound ;
            property OnError         : TCoverDownloaderStatusEvent       read fOnError          write fOnError         ;
            property OnBeforeSave    : TCoverDownloaderBeforeSaveEvent   read fOnBeforeSave     write fOnBeforeSave    ;

            property SavedFile: String read fSavedFile;

            function DownloadCover(aArtist, aAlbum, aDirectory: String): Boolean;
    end;


implementation

uses  HtmlHelper;


constructor TCoverDownloader.Create;
begin
    inherited create;
    fIDHttp := TIdHttp.Create;
    fDataStream := TMemoryStream.Create;

    fIDHttp.ConnectTimeout:= 5000;
    fIDHttp.ReadTimeout:= 5000;
    fIDHttp.Request.UserAgent := 'Mozilla/3.0';
    fIDHttp.HTTPOptions :=  [];
end;

destructor TCoverDownloader.Destroy;
begin
    fIDHttp.Free;
    fDataStream.Free;
    inherited;
end;


function TCoverDownloader.DownloadCover(aArtist,
  aAlbum, aDirectory: String): Boolean;
var xmlData, CoverURL: AnsiString;
    DownloadComplete: Boolean;
begin

    xmlData := QueryLastFMCoverXML(aArtist, aAlbum); // Get XML-File from API
    if assigned(fOnXMLComplete) then
        fOnXMLComplete(self, 'XML-Daten geladen ...');

    CoverURL := GetBestCoverUrlFromXML(xmlData);      // Get a Cover-URL from the XML-File
    if assigned(fOnCoverURLFound) then
        fOnCoverURLFound(self, 'Cover gefunden. Download startet ...');

    DownloadComplete := DownloadBestCoverToStream(CoverURL);    // download the Cover from the URL

    if DownloadComplete then
        DownloadComplete := SavePicStreamToFile(aDirectory);   // save downloaded picture to a file in the directory

    result := DownloadComplete;
end;

{
    --------------------------------------------------------
    QueryLastFMCoverXML:
    - download the XML-Reply from the LastFM API
      save the reply in Self.fXMLData
    --------------------------------------------------------
}
function TCoverDownloader.QueryLastFMCoverXML(aArtist, aAlbum: String): AnsiString;
var url: UTF8String;
begin
    url := 'http://ws.audioscrobbler.com/2.0/?method=album.getinfo'
        + '&api_key=' + api_key
        + '&artist=' + StringToURLStringAnd(AnsiLowerCase(aArtist))
        + '&album='  + StringToURLStringAnd(AnsiLowerCase(aAlbum));
    try
        result := fIDHttp.Get(url);
    except
        on E: Exception do
        begin
            result := '';
            if assigned(fOnError) then
                fOnError(self, E.Message);
        end;
    end;
end;

{
    --------------------------------------------------------
    GetBestCoverUrlFromXML:
    - Parse the fXMLData and get a Cover URL
      save the URL in Self.fBestCoverURL
    --------------------------------------------------------
}
function TCoverDownloader.GetBestCoverUrlFromXML(aXMLData: AnsiString): AnsiString;
var s, e: Integer;
    offset: Integer;
begin
    if aXMLData = '' then
        result := ''
    else
    begin
        s := Pos('<image size="extralarge">', aXMLData);
        offset := length('<image size="extralarge">');

        if s = 0 then
        begin
            s := Pos('<image size="large">', aXMLData);
            offset := length('<image size="large">');
        end;

        if s > 0 then
        begin
            e := PosEx('</image>', aXMLData, s);
            result := Copy(aXMLData, s + offset, e - (s + offset));
        end else
        begin
            result := '';
            if assigned(fOnError) then
                fOnError(self, 'Keine Cover-URL gefunden.');
        end;
    end;
end;


{
    --------------------------------------------------------
    DownloadBestCoverToStream:
    - Download Cover from fBestCoverURL
    --------------------------------------------------------
}
function TCoverDownloader.DownloadBestCoverToStream(aCoverURL: AnsiString): Boolean;
begin
    if aCoverURL <> '' then
    begin
        fDataStream.Clear;
        try
            fIDHttp.Get(aCoverURL, fDataStream);
            result := True;
        except
            on E: Exception do
            begin
                fDataStream.Clear;
                result := False;
                fDataType := ptNone;
                if assigned(fOnError) then
                    fOnError(self, E.Message);
            end;
        end;

        if AnsiEndsText('.jpg', aCoverURL) then
            fDataType := ptJPG
        else
        if AnsiEndsText('.png', aCoverURL) then
            fDataType := ptPNG
        else
        if AnsiEndsText('.jpeg', aCoverURL) then
            fDataType := ptJPG
        else
        begin
            fDataType := ptNone;
            fDataStream.Clear;
            result := False;
            if assigned(fOnError) then
                fOnError(self, 'Unbekannter Bildtyp. Vorgang abgebrochen.');
        end;
    end else
    begin
        result := False;
        fDataType := ptNone;
        fDataStream.Clear;
        if assigned(fOnError) then
            fOnError(self, 'Keine Cover-URL gefunden.');
    end;
end;

{
    --------------------------------------------------------
    SavePicStreamToFile:
    - Save the downloaded Stream to a file
    --------------------------------------------------------
}
function TCoverDownloader.SavePicStreamToFile(aDirectory: String): boolean;
var NewCoverFilename: String;
begin
    case fDataType of
        ptJPG:  NewCoverFilename := aDirectory + 'front (CovernatorDownload).jpg';
        ptPNG:  NewCoverFilename := aDirectory + 'front (CovernatorDownload).png';
    else
        NewCoverFilename := '';
    end;

    if assigned(fOnBeforeSave) then
        fOnBeforeSave(self, fDataType, NewCoverFilename );

    if NewCoverFilename <> '' then
    begin
        try
            fDataStream.SaveToFile(NewCoverFilename);
            fSavedFile := NewCoverFilename;
            result := True;
        except
            on E: Exception do
            begin
                result := False;
                if assigned(fOnError) then
                    fOnError(self, E.Message);
            end;
        end;
    end else
        result := False;
end;



end.
